/**
 * 
 */
package de.consist.bmu.rule.impl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;
import jakarta.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.DocumentController;
import de.consist.bmu.rule.MeldungTyp;
import de.consist.bmu.rule.MeldungTyp.FehlerKlasse;
import de.consist.bmu.rule.MeldungTyp.FehlerStufe;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.RuleSet;
import de.consist.bmu.rule.RuleSetResult;
import de.consist.bmu.rule.def.MeldungTypImpl;
import de.consist.bmu.rule.def.RuleDefImpl;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.error.BMUParseException;
import de.consist.bmu.rule.schema.BMUMessageTypeImpl;
import de.consist.bmu.rule.util.ByteUtils;
import de.consist.bmu.rule.util.DateUtils;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "bmudok", propOrder = { "_bmuMessageType" })
@XmlRootElement(name = "BMUDokument")
public final class BMUDokumentImpl implements BMUDokument, Serializable {

    /** */
    private static final long serialVersionUID = 1L;

    private static final Log LOGGER = LogFactory.getLog(BMUDokumentImpl.class);

    private static final MeldungTyp MELDUNG_PARSE_ERROR = new MeldungTypImpl(
            FehlerKlasse.KEINXML, FehlerStufe.FATAL,
            "Fehler beim Parsen des Dokuments: ", "PB_X0",
            "Das Dokument kann nicht verarbeitet werden");

    private static final RuleDef RULEDEF_PARSE_ERROR = new RuleDefImpl(
            "ID_PARSE_ERROR", "ParseError", null, true, new Date(0L), true,
            MELDUNG_PARSE_ERROR, null, new Date(0L), false);

    private static boolean _validateParse = true;

    @XmlElement(name = "MessageType", type = BMUMessageTypeImpl.class)
    private BMUMessageType _bmuMessageType;

    private transient int _size;

    private transient Document _doc;

    /**
     * Default Konstruktor fuer JAXB.
     */
    public BMUDokumentImpl() {
    }

    /**
     * Adapter fuer JAXB.
     */
    public static class Adapter extends XmlAdapter<BMUDokumentImpl, BMUDokument> {
        /**
         * {@inheritDoc}
         */
        @Override
        public final BMUDokument unmarshal(BMUDokumentImpl bmuDokImpl) {
            return bmuDokImpl;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public final BMUDokumentImpl marshal(BMUDokument bmuDok) {
            return (BMUDokumentImpl) bmuDok;
        }
    }

    private BMUDokumentImpl(Document doc, BMUMessageType bmuMessageType, int size) {
        _doc = doc;
        _bmuMessageType = bmuMessageType;
        _size = size;
    }

    private static void parseError(RuleSet ruleSet, List<RuleResult> ruleResultList, String msg,
            RuleSetResult.Status status) throws BMUParseException {
        String bezugMessage = "Prfzeitpunkt: "
                + DateUtils.toDateTimeString(new Date());
        RuleResult bezugResult = new RuleResultImpl(ruleSet.getRuleDefBezugError(), 1, bezugMessage);
        RuleSetResult ruleSetResult = new RuleSetResultImpl(null, null, ruleResultList, bezugResult, FehlerStufe.FATAL,
                status);
        throw new BMUParseException(msg, ruleSetResult);
    }

    /**
     * @param is
     *            InputStream
     * @param ruleSet
     *            Das RuleSet
     * @return BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parse(InputStream is, RuleSet ruleSet) throws BMUParseException {
        BMUDokument bmuDok = null;
        List<RuleResult> ruleResultList = null;
        try {
            byte[] data = ByteUtils.readFromStream(is);
            bmuDok = parse(new ByteArrayInputStream(data), data.length, ruleSet);
        } catch (BMUException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("parse error: " + e.getMessage());
            }
            RuleResult ruleResult = new RuleResultImpl(RULEDEF_PARSE_ERROR, 1, e.getMessage());
            ruleResultList = new ArrayList<RuleResult>();
            ruleResultList.add(ruleResult);
            parseError(ruleSet, ruleResultList, e.getMessage(), RuleSetResult.Status.PARSE_ERROR);
        }
        return bmuDok;
    }

    /**
     * @param is
     *            InputStream
     * @param size
     *            Die Anzahl der Bytes des Dokuments
     * @param ruleSet
     *            Das RuleSet
     * @return BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parse(InputStream is, int size, RuleSet ruleSet) throws BMUParseException {
        Document doc = null;
        BMUMessageType msgType = null;
        List<RuleResult> ruleResultList = null;
        try {
            doc = DocumentController.parse(is);
            boolean isOSCIQuittung = false;
            try {
                isOSCIQuittung = XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
                        "local-name(.)='Envelope'");
            } catch (XPathExpressionException e) {
                LOGGER.warn("error evaluating XPath 'local-name(.)='Envelope'': " + e.getMessage());
            }
            if (!isOSCIQuittung && _validateParse) {
//                if (SchemaValidator.isUsingVersion1_04a()) {
                    RuleImplSchemaValidation104a ruleSchemaValidation = (RuleImplSchemaValidation104a) ruleSet
                            .getRuleById("ID_042");
                    ruleResultList = ruleSchemaValidation.executeInternal(doc);
//                } else {
//                    RuleImplSchemaValidation ruleSchemaValidation = (RuleImplSchemaValidation) ruleSet
//                            .getRuleById("ID_002");
//                    ruleResultList = ruleSchemaValidation.executeInternal(doc);
//                }
                if (ruleResultList.size() > 0) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("schema error");
                    }
                    parseError(ruleSet, ruleResultList, "Schema validation failed", RuleSetResult.Status.SCHEMA_ERROR);
                }
            }
            msgType = BMUMessageTypeImpl.getMessageType(doc);
        } catch (BMUException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("parse error: " + e.getMessage());
            }
            RuleResult ruleResult = new RuleResultImpl(RULEDEF_PARSE_ERROR, 1, e.getMessage());
            ruleResultList = new ArrayList<RuleResult>();
            ruleResultList.add(ruleResult);
            parseError(ruleSet, ruleResultList, e.getMessage(), RuleSetResult.Status.PARSE_ERROR);
        }

        return new BMUDokumentImpl(doc, msgType, size);
    }

    /**
     * @param file
     *            File
     * @param ruleSet
     *            Das RuleSet
     * @return BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parse(File file, RuleSet ruleSet) throws BMUParseException {
        BMUDokument bmuDok = null;
        List<RuleResult> ruleResultList = null;
        try {
            bmuDok = parse(new FileInputStream(file), (int) file.length(), ruleSet);
        } catch (FileNotFoundException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("file not found: " + file.getAbsolutePath());
            }
            RuleResult ruleResult = new RuleResultImpl(RULEDEF_PARSE_ERROR, 1, e.getMessage());
            ruleResultList = new ArrayList<RuleResult>();
            ruleResultList.add(ruleResult);
            parseError(ruleSet, ruleResultList, e.getMessage(), RuleSetResult.Status.PARSE_ERROR);
        }
        return bmuDok;
    }

    /**
     * {@inheritDoc}
     */
    public Document getDocument() {
        return _doc;
    }

    /**
     * {@inheritDoc}
     */
    public BMUMessageType getMessageType() {
        return _bmuMessageType;
    }

    /**
     * {@inheritDoc}
     */
    public int getSize() {
        return _size;
    }

    private void writeObject(ObjectOutputStream oos) throws IOException {
        LOGGER.trace("<BMUDokumentImpl.writeObject>");
        oos.defaultWriteObject();

        // Write/save additional fields
        writeDocument(oos, _doc);
    }

    private void readObject(ObjectInputStream ois) throws IOException,
            ClassNotFoundException {
        LOGGER.trace("<BMUDokumentImpl.readObject>");
        ois.defaultReadObject();

        // Read/initialize additional fields
        _doc = readDocument(ois);
        // try {
        // _bmuMessageType = BMUMessageTypeImpl.getMessageType(_doc);
        // } catch (BMUException e) {
        // LOGGER.error("<BMUDokumentImpl.readObject> Fehler", e);
        // throw new IOException(e.getMessage(), e);
        // }
    }

    private void writeDocument(ObjectOutputStream oos, Document doc)
            throws IOException {
        if (doc == null) {
            oos.writeInt(0);
        } else {
            byte[] b1 = null;
            try {
                b1 = DocumentController.serialize(doc);
            } catch (BMUException e) {
                LOGGER.error("<BMUDokumentImpl.writeDocument> Fehler", e);
                throw new IOException(e.getMessage());
            }
            oos.writeInt(b1.length);
            if (b1.length > 0) {
                oos.write(b1);
            } else {
                LOGGER.error("Unerwarteter Zustand: Lnge des Bytearray kleiner 1 ("
                        + Integer.toString(b1.length) + ")");
            }
        }
    }

    private Document readDocument(ObjectInputStream ois) throws IOException {
        int len = ois.readInt();
        if (len > 0) {
            byte[] buf = new byte[len];
            ois.readFully(buf);
            _size = buf.length;
            try {
                return DocumentController.parse(buf);
            } catch (BMUException e) {
                LOGGER.error("<BMUDokumentImpl.readDocument> Fehler", e);
                throw new IOException(e.getMessage());
            }
        }
        return null;
    }

    /**
     * @return the _validateParse
     */
    public static boolean isValidateParse() {
        return _validateParse;
    }

    /**
     * @param validateParse
     *            the validateParse to set
     */
    public static void setValidateParse(boolean validateParse) {
        BMUDokumentImpl._validateParse = validateParse;
    }
}
