/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2013-06-28 08:24:11 +0000 (Fri, 28 Jun 2013) $ $Rev: 1091 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
public final class RuleImplListIndex extends RuleImpl {

    private static final long serialVersionUID = 1L;

    static class NameXPath {
        private String _name;
        private String _xPath;
        NameXPath(String name, String xPath) {
            _name = name;
            _xPath = xPath;
        }
    };
    
    private static final NameXPath[] XPATH_BGS_INDEX = {
            new NameXPath("Fehler", "bgs:Fehler"), 
            new NameXPath("Sortiment", "bgs:Sortiment"),
            new NameXPath("UNSNummer", "bgs:UNSNummer"),
            new NameXPath("WeitereAbfgallschluessel", "bgs:WeitereAbfallschluessel"),
            new NameXPath("PCBFraktion", "bgs:PCBFraktion") };
    private static final NameXPath[] XPATH_ENS_INDEX = {
            new NameXPath("BImSchG", "en:BImSchG"),
            new NameXPath("Kreis", "en:Kreis"),
            new NameXPath("Bundesland", "en:Bundesland"),
            new NameXPath("Fehler", "en:Fehler"),
            new NameXPath("Nebenbestimmung BB", "en:Nebenbestimmung[parent::en:BB]"),
            new NameXPath("Begruendung BB", "en:Begruendung[parent::en:BB]"),
            new NameXPath("Adressat BB", "en:Adressat[ancestor::en:BB]"),
            new NameXPath("Nebenbestimmung ANO", "en:Nebenbestimmung[parent::en:Anordnung]"),
            new NameXPath("Begruendung ANO", "en:Begruendung[parent::en:Anordnung]"),
            new NameXPath("Adressat ANO", "en:Adressat[ancestor::en:Anordnung]") };
    private static final NameXPath[] XPATH_FR_INDEX = {
            new NameXPath("Abfall", "en:Abfall"),
            new NameXPath("Nebenbestimmung", "en:Nebenbestimmung[parent::en:BB]"),
            new NameXPath("Begruendung", "en:Begruendung[parent::en:BB]"),
            new NameXPath("Adressat", "en:Adressat[ancestor::en:BB]") };
    private static final NameXPath[] XPATH_EGF_INDEX = { 
            new NameXPath("Bundesland", "en:Bundesland") };

    private static final Log LOGGER = LogFactory
            .getLog(RuleImplListIndex.class);

    /**
     * @param ruleDef RuleDef
     */
    public RuleImplListIndex(RuleDef ruleDef) {
        super(ruleDef);
    }

    private int getMaxValue(NodeList nl) {
        int maxValue = 0;
        for (int i = 0; i < nl.getLength(); i++) {
            int indexValue = Integer.parseInt(nl.item(i).getTextContent());
            if (indexValue > maxValue) {
                maxValue = indexValue;
            }
        }
        return maxValue;
    }

    private List<RuleResult> checkIndex(NameXPath nx, Document doc, int index)
            throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        String xPathIndex0 = "count(/descendant::" + nx._xPath + "[@lib:Index=0]) > 0";
        try {
            if (XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
                    xPathIndex0)) {
                ruleResultList.add(new RuleResultImpl(getRuleDef(), index++, nx._name
                        + " mit Index 0"));
            }
            NodeList nl = XPathFassade.getInstance().evaluateNodeList(doc,
                    "/descendant::" + nx._xPath + "/@lib:Index");
            int maxValue = getMaxValue(nl);
            for (int i = 1; i <= maxValue; i++) {
                String indexAttr = "[@lib:Index=" + i + "]";
                String indexXPath = "/descendant::" + nx._xPath + indexAttr;
                if (!XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
                        indexXPath)) {
                    ruleResultList.add(new RuleResultImpl(getRuleDef(), index++, nx._name
                            + " mit Index " + i + " fehlt"));
                } else {
                    String multipleIndexXPath = indexXPath + "/preceding-sibling::"
                            + nx._xPath + indexAttr;
                    //System.out.println(multipleIndexXPath);
                    if (XPathFassade.getInstance().evalBool(
                            doc.getDocumentElement(), multipleIndexXPath)) {
                        // System.out.println(name + " mit Index " + i +
                        // " gibt es auf dieser Ebene schon");
                        ruleResultList.add(new RuleResultImpl(getRuleDef(), index++, nx._name
                                + " mit Index " + i
                                + " gibt es auf dieser Ebene schon"));
                    }
                }
            }
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Prfen", e);
        }
        return ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = bmuDok.getDocument();
        BMUMessageType msgType = bmuDok.getMessageType();
        int index = 1;
        NameXPath[] nxp = null;
        switch (msgType.getEnumType()) {
        case ENSNDokument:
            nxp = XPATH_ENS_INDEX;
            break;
        case BGSDokument:
            nxp = XPATH_BGS_INDEX;
            break;
        case EGFDokument:
            nxp = XPATH_EGF_INDEX;
            break;
        case FRDokument:
            nxp = XPATH_FR_INDEX;
            break;
        default:
            LOGGER.warn("unexpected message-type: " + msgType.getEnumType());
        }
        if (nxp != null) {
            for (NameXPath nx : nxp) {
                List<RuleResult> rrl = checkIndex(nx, doc, index); 
                index += rrl.size();
                ruleResultList.addAll(rrl);
            }
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        StringBuilder builder = new StringBuilder();
        builder.append("Rollbereiche in Nachweisdokumenten:");
        builder.append(RuleImpl.NL);
        for (NameXPath nx : XPATH_ENS_INDEX) {
            builder.append(" Name: ");
            builder.append(nx._name);
            builder.append(", XPath: ");
            builder.append("/descendant::" + nx._xPath + "/@lib:Index");
            builder.append(RuleImpl.NL);
        }
        builder.append(", Rollbereiche in Begleitscheinen:");
        builder.append(RuleImpl.NL);
        for (NameXPath nx : XPATH_BGS_INDEX) {
            builder.append("Name: ");
            builder.append(nx._name);
            builder.append(", XPath: ");
            builder.append("/descendant::" + nx._xPath + "/@lib:Index");
            builder.append(RuleImpl.NL);
        }
        builder.append(", Rollbereiche in EGFDokumenten:");
        builder.append(RuleImpl.NL);
        for (NameXPath nx : XPATH_EGF_INDEX) {
            builder.append("Name: ");
            builder.append(nx._name);
            builder.append(", XPath: ");
            builder.append("/descendant::" + nx._xPath + "/@lib:Index");
            builder.append(RuleImpl.NL);
        }
        builder.append(", Rollbereiche in Freistellungen:");
        builder.append(RuleImpl.NL);
        for (NameXPath nx : XPATH_FR_INDEX) {
            builder.append("Name: ");
            builder.append(nx._name);
            builder.append(", XPath: ");
            builder.append("/descendant::" + nx._xPath + "/@lib:Index");
            builder.append(RuleImpl.NL);
        }
        return builder.toString();
    }

}
