/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2025-05-15 08:56:28 +0000 (Thu, 15 May 2025) $ $Rev: 1865 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.schema.Namespace;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
public final class RuleImplTextLength extends RuleImpl {

    private static final long serialVersionUID = 1L;

    private static final int MAX_LENGTH = 255;

    private static final Log LOGGER = LogFactory
            .getLog(RuleImplTextLength.class);

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) in Nachweisen (ENS).
     */
    private static final String[] XPATH_ENS_TEXT_LENGTH = { "en:Erklaerung/en:Zusatz" };

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) in Registrierungsantraegen
     * (RGA).
     */
    private static final String[] XPATH_MT_TEXT_LENGTH = {
            "msg:Mitteilung/msg:Thema", "msg:Mitteilung/msg:Bezug" };

    private static final String XPATH_BGS_SORTIMENT = "/descendant::bgs:Sortiment[parent::bgs:AltholzV]";

    /**
     * @param ruleDef
     *            RuleDef
     */
    public RuleImplTextLength(RuleDef ruleDef) {
        super(ruleDef);
    }

    private List<RuleResult> evalTextLength(Document doc, String[] xPathes)
            throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        int index = 1;
        for (String xpFT : xPathes) {
            try {
                if (XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
                        "string-length(normalize-space(/descendant::" + xpFT + "))>255")) {
                    ruleResultList.add(new RuleResultImpl(getRuleDef(), index++,
                            xpFT));
                }
            } catch (XPathExpressionException e) {
                throw new BMUException("Fehler beim Prfen", e);
            }
        }
        return ruleResultList;
    }

    private List<RuleResult> evalBGSSortiment(Document doc) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        NodeList nl;
        try {
            nl = XPathFassade.getInstance().evaluateNodeList(doc,
                    XPATH_BGS_SORTIMENT);
            int maxIndex = 0;
            for (int i = 0; i < nl.getLength(); i++) {
                Element e = (Element) nl.item(i);
                String indexString = e.getAttributeNS(Namespace.TypenBibliothek.getUri(), "Index");
                int index = Integer.parseInt(indexString);
                if (index > maxIndex) {
                    maxIndex = index;
                }
            }
            int codeIndex = 1;
            for (int i = 0; i <= maxIndex; i++) {
                String xPath = "/descendant::bgs:Sortiment[@lib:Index=" + i + "][last()]/bgs:Informationen";
                String info = XPathFassade.getInstance().evaluate(doc, xPath);
                if (info.length() > MAX_LENGTH) {
                    ruleResultList.add(new RuleResultImpl(getRuleDef(), codeIndex++,
                            xPath));
                }
            }
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Prfen", e);
        }
        return ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = bmuDok.getDocument();
        BMUMessageType msgType = bmuDok
                .getMessageType();
        switch (msgType.getEnumType()) {
        case ENSNDokument:
            ruleResultList.addAll(evalTextLength(doc, XPATH_ENS_TEXT_LENGTH));
            break;
        case BGSDokument:
            ruleResultList.addAll(evalBGSSortiment(doc));
            break;
        case Mitteilung:
            ruleResultList.addAll(evalTextLength(doc, XPATH_MT_TEXT_LENGTH));
            break;
        default:
            LOGGER.warn("unexpected message-type: " + msgType.getEnumType());
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        StringBuilder builder = new StringBuilder();
        builder.append("XPath fr die Ermittlung von Sortimenten in Begleitscheinen: ");
        builder.append(XPATH_BGS_SORTIMENT);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr die Prfung der Lnge des Elements 'Informationen' im 'Sortiment' mit Index X: ");
        builder.append("string-length(/descendant::bgs:Sortiment[@lib:Index=X][last()]/bgs:Informationen)>255");
        builder.append(RuleImpl.NL);
        builder.append("XPath fr die Prfung von Texten in Nachweisen: ");
        builder.append("string-length(/descendant::");
        builder.append(XPATH_ENS_TEXT_LENGTH[0]);
        builder.append(")>255");
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von Texten in Mitteilungen: ");
        for (String xPath : XPATH_MT_TEXT_LENGTH) {
            builder.append("string-length(/descendant::");
            builder.append(xPath);
            builder.append(")>255, ");
        }
        return builder.toString();
    }
}
