/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2022-03-09 10:01:19 +0000 (Wed, 09 Mar 2022) $ $Rev: 1700 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.schema;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * LSResourceResolver Implementation f&uuml;r die Benutzung von
 * {@link SchemaValidator}. Benutzte Resourcen werden in einer internen Map
 * gehalten.
 * 
 * @author srossbroich
 */
public class BMUSchemaResourceResolver implements LSResourceResolver {

    private static final Log LOGGER = LogFactory
            .getLog(BMUSchemaResourceResolver.class);

    /**
     * Ressource Map URI -> XSD-File Alias -> XSD-File.
     */
    private static final Map<String, String> RESOURCE_MAPPING;

    static {
        RESOURCE_MAPPING = new HashMap<String, String>();
        
        RESOURCE_MAPPING.put("http://www.w3.org/2001/XMLSchema.dtd",
                "2001/XMLSchema.dtd");
        
//        RESOURCE_MAPPING.put("urn:de:bmu:eanv:ServiceModul:0:4", "ZertifikatsPruefung.xsd");
        
        // eudin/standard/common/
        RESOURCE_MAPPING
                .put("urn:un:unece:uncefact:data:documentation:CoreComponentTypeSchemaModule:2:1",
                        "Eudin_XSD/standard/common/CCTS_CCT_SchemaModule-2.1.xsd");
        RESOURCE_MAPPING.put(
                "urn:un:unece:uncefact:codelist:specification:54217:2001",
                "Eudin_XSD/standard/common/CodeList_CurrencyCode_ISO_7_04.xsd");
        RESOURCE_MAPPING.put(
                "urn:un:unece:uncefact:codelist:specification:5639:1988",
                "Eudin_XSD/standard/common/CodeList_LanguageCode_ISO_7_04.xsd");
        RESOURCE_MAPPING
                .put("urn:un:unece:uncefact:codelist:specification:IANAMIMEMediaTypes:2003",
                        "Eudin_XSD/standard/common/CodeList_MIMEMediaTypeCode_IANA_7_04.xsd");
        RESOURCE_MAPPING.put(
                "urn:un:unece:uncefact:codelist:specification:66411:2001",
                "Eudin_XSD/standard/common/CodeList_UnitCode_UNECE_7_04.xsd");
        RESOURCE_MAPPING
                .put("urn:un:unece:uncefact:data:specification:QualifiedDataTypeSchemaModule:2:1",
                        "Eudin_XSD/standard/common/QualifiedDataTypeSchemaModule-2.1.xsd");
        RESOURCE_MAPPING
                .put("urn:un:unece:uncefact:data:specification:ReusableAggregateBusinessInformationEntitySchemaModule:2:1",
                        "Eudin_XSD/standard/common/ReusableAggregateBusinessInformationEntitySchemaModule-2.1.xsd");
        RESOURCE_MAPPING
                .put("urn:un:unece:uncefact:data:specification:UnqualifiedDataTypesSchemaModule:2:1",
                        "Eudin_XSD/standard/common/UnqualifiedDataTypeSchemaModule-2.1.xsd");

        // eudin/user/common/
        RESOURCE_MAPPING.put("usr:QualifiedDataTypeSchemaModule:2:1",
                "Eudin_XSD/user/common/QualifiedDataTypeSchemaModule-2.1.xsd");
        RESOURCE_MAPPING
                .put("usr:ReusableAggregateBusinessInformationEntitySchemaModule:2:1",
                        "Eudin_XSD/user/common/ReusableAggregateBusinessInformationEntitySchemaModule-2.1.xsd");

        // eudin/user/maindoc/
        RESOURCE_MAPPING
                .put("usr:EUDINCertificateOfWasteReceiptDocument:2:1",
                        "Eudin_XSD/user/maindoc/EUDINCertificateOfWasteReceiptDocument-2.1.xsd");
        RESOURCE_MAPPING
                .put("usr:EUDINCertificateOfWasteRecoveryDisposalDocument:2:1",
                        "Eudin_XSD/EUDINCertificateOfWasteRecoveryDisposalDocument-2.1.xsd");
        RESOURCE_MAPPING
                .put("usr:EUDINConfirmationOfMessageReceipt:2:1",
                        "Eudin_XSD/user/maindoc/EUDINConfirmationOfMessageReceipt-2.1.xsd");
        RESOURCE_MAPPING.put("usr:EUDINWasteMovementDocument:2:1",
                "Eudin_XSD/user/maindoc/EUDINWasteMovementDocument-2.1.xsd");
        RESOURCE_MAPPING.put("usr:EUDINWasteTransportStatement:2:1",
                "Eudin_XSD/user/maindoc/EUDINWasteTransportStatement-2.1.xsd");

        // da/
        RESOURCE_MAPPING.put("DA_ParameterlisteTeil_3_Anhang_E.xsd",
                "DA_XSD/DA_ParameterlisteTeil_3_Anhang_E.xsd");
        RESOURCE_MAPPING.put("uri:BMU_Waste_Interface/DA",
                "DA_XSD/DA_Teil_2_Anhang_E.xsd");
        RESOURCE_MAPPING.put("uri:BMU_Waste_Interface/DATypen",
                "DA_XSD/DA_Typen_Teil_1-Anhang_E.xsd");
        
        // AbfAEV
        RESOURCE_MAPPING.put("uri:AnzeigeErlaubnis/AbfAEV",
                "AbfAEV/FreieXMLStruktur-AbfAEV.xsd");
        RESOURCE_MAPPING.put("uri:AnzeigeErlaubnis/Kopie_TypBibliothek",
                "AbfAEV/Kopie_TypBibliothek.xsd");
        RESOURCE_MAPPING.put("uri:AnzeigeErlaubnis/Kopie_Kataloge",
                "AbfAEV/Kopie_Kataloge.xsd");
        
        // ErzeugerStammdaten
        RESOURCE_MAPPING.put("uri:ErzeugerStammdaten/ErzStamm",
                "ErzStamm/FreieXMLStrukturErzStamm.xsd");
        RESOURCE_MAPPING.put("uri:ErzeugerStammdaten/ErzStammTypBibliothek",
                "ErzStamm/ErzStammTypBibliothek.xsd");
        RESOURCE_MAPPING.put("uri:ErzeugerStammdaten/ErzStammKataloge",
                "ErzStamm/ErzStammKataloge.xsd");
        
        // waste shipment guidelines 11
        RESOURCE_MAPPING.put("Waste-WG-2018-Message", "waste-shipment-guidelines-11/message.xsd");
        RESOURCE_MAPPING.put("codelists.xsd", "waste-shipment-guidelines-11/codelists.xsd");
        RESOURCE_MAPPING.put("basetypes.xsd", "waste-shipment-guidelines-11/basetypes.xsd");
        RESOURCE_MAPPING.put("annex7.xsd", "waste-shipment-guidelines-11/annex7.xsd");
        RESOURCE_MAPPING.put("notification.xsd", "waste-shipment-guidelines-11/notification.xsd");
        RESOURCE_MAPPING.put("statement.xsd", "waste-shipment-guidelines-11/statement.xsd");
        RESOURCE_MAPPING.put("wastemovement.xsd", "waste-shipment-guidelines-11/wastemovement.xsd");
    }

    /**
     * Basispfad zu den Schema-Dateien.
     */
    private final String _schemaFilePath;

    /**
     * Default Konstruktor.
     * 
     * @param aSchemaFilePath
     *            Basispfad zu den Schemadateien
     */
    public BMUSchemaResourceResolver(final String aSchemaFilePath) {
        this._schemaFilePath = aSchemaFilePath;
    }

    /**
     * Ruft resolveResource(String type, String namespaceURI, String publicId,
     * String systemId, String baseURI)) mit type und namespaceURI sind null
     * gesetzt.
     * 
     * @param publicId
     *            PUBLIC id
     * @param systemId
     *            SYSTEM id
     * @param baseURI
     *            baseURI
     * @return ressource
     * 
     */
    public final LSInput resolveResource(final String publicId,
            final String systemId, final String baseURI) {
        return resolveResource(null, null, publicId, systemId, baseURI);
    }

    /**
     * Sucht die gesuchte Resource anhang der lokal verf&uuml;gbaren Resourcen.
     * 
     * @param type
     *            - The type of the resource being resolved. For XML [XML 1.0]
     *            resources (i.e. entities), applications must use the value
     *            "http://www.w3.org/TR/REC-xml". For XML Schema [XML Schema
     *            Part 1] , applications must use the value
     *            "http://www.w3.org/2001/XMLSchema".
     * @param namespaceURI
     *            - The namespace of the resource being resolved, e.g. the
     *            target namespace of the XML Schema [XML Schema Part 1] when
     *            resolving XML Schema resources.
     * @param publicId
     *            - The public identifier of the external entity being
     *            referenced, or null if no public identifier was supplied or if
     *            the resource is not an entity.
     * @param systemId
     *            - The system identifier, a URI reference [IETF RFC 2396], of
     *            the external resource being referenced, or null if no system
     *            identifier was supplied.
     * @param baseURI
     *            - The absolute base URI of the resource being parsed, or null
     *            if there is no base URI.
     * @return A ByteStream connected to the resource
     * 
     * 
     */
    public final LSInput resolveResource(final String type,
            final String namespaceURI, final String publicId,
            final String systemId, final String baseURI) {
        LOGGER.debug("trying to resolve: publicId=" + publicId + ", systemId=" + systemId + ", baseURI=" + baseURI);
    	LSInput lsInput = new LSInputImpl();
        URL url = null;
        if (systemId != null) {
            if (RESOURCE_MAPPING.containsKey(systemId)) {
                url = getClass().getResource(
                        this._schemaFilePath + RESOURCE_MAPPING.get(systemId));
            } else {
                url = getClass().getResource(this._schemaFilePath + systemId);
            }
        } 
        if (url == null && publicId != null && RESOURCE_MAPPING.containsKey(publicId)) {
            url = getClass().getResource(
                    this._schemaFilePath + RESOURCE_MAPPING.get(publicId));
        } 
        if (url == null && namespaceURI != null
                && RESOURCE_MAPPING.containsKey(namespaceURI)) {
            url = getClass().getResource(
                    this._schemaFilePath + RESOURCE_MAPPING.get(namespaceURI));
        }
        if (url == null) {
            LOGGER.error("unable to resolve: systemId=" + systemId
                    + ", pubicId=" + publicId + ", namespaceURI="
                    + namespaceURI);
        } else {

            try {
                lsInput.setByteStream(url.openStream());
                LOGGER.debug("resolved: " + url);
            } catch (IOException e) {
                LOGGER.error("Error resolving resource: " + url.getPath(), e);
            }
        }
        return lsInput;
    }

}
