/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-07 12:06:34 +0000 (Wed, 07 Feb 2024) $ $Rev: 1793 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import de.consist.bmu.rule.error.BMUException;

/**
 * Hilfsroutinen fr XML-Dokumente.
 * 
 * @author jannighoefer
 */
public final class XmlUtils {

	private static final Log LOGGER = LogFactory.getLog(XmlUtils.class);

	private XmlUtils() {
	}

	/**
	 * @param doc
	 *            Document
	 * @param filename
	 *            String
	 * @throws BMUException
	 *             BMUException
	 */
	public static void writeToFile(Document doc, String filename) throws BMUException {
		XmlUtils.writeToFile(doc, filename, null);
	}

	/**
	 * @param doc
	 *            Document
	 * @param filename
	 *            String
	 * @param encoding
	 *            String
	 * @throws BMUException
	 *             BMUException
	 */
	public static void writeToFile(Document doc, String filename, String encoding) throws BMUException {
		File f = new File(filename);
		try {
			FileOutputStream fos = new FileOutputStream(f);
			XmlUtils.writeToStream(doc, fos, encoding);
		} catch (Exception ex) {
			XmlUtils.LOGGER.error("error writing to file: " + f.getAbsolutePath(), ex);
			throw new BMUException("error writing to file: " + f.getAbsolutePath(), ex);
		}
	}

	/**
	 * @param doc
	 *            Document
	 * @param outStream
	 *            OutputStream
	 * @param encoding
	 *            String
	 * @throws BMUException
	 *             BMUException
	 */
	public static void writeToStream(Document doc, OutputStream outStream, String encoding) throws BMUException {
		TransformerFactory tFactory = TransformerFactory.newInstance();
		try {
			DOMSource source = new DOMSource(doc);
			Transformer transformer = tFactory.newTransformer();
			StreamResult result = null;
			if (encoding != null) {
				transformer.setOutputProperty(OutputKeys.ENCODING, encoding);
				result = new StreamResult(new OutputStreamWriter(outStream, encoding));
			} else {
				result = new StreamResult(outStream);
			}
			transformer.transform(source, result);
		} catch (Exception ex) {
			XmlUtils.LOGGER.error("error writing to stream", ex);
			throw new BMUException("error writing to stream", ex);
		}
	}

	/**
	 * Schreibt das Document UTF-8-Kodiert in ein Bytearray.
	 * 
	 * @param doc
	 *            Document
	 * @return byte[]
	 * @throws BMUException
	 *             BMUException
	 */
	public static byte[] serialize(Document doc) throws BMUException {
		byte[] retVal = null;
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		XmlUtils.writeToStream(doc, bos, "UTF-8");
		retVal = bos.toByteArray();
		return retVal;
	}

	/**
	 * @param filename
	 *            String
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document readFromFile(String filename) throws BMUException {
		return XmlUtils.readFromFile(filename, null);
	}

	/**
	 * @param name
	 *            String
	 * @param encoding
	 *            String
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document readFromResource(String name, String encoding) throws BMUException {
		InputStream is = XmlUtils.class.getResourceAsStream(name);
		if (is == null) {
			XmlUtils.LOGGER.error("error loading resource: " + name);
			throw new BMUException("error loading resource: " + name);
		}
		return XmlUtils.readFromStream(is);
	}

	/**
	 * @param filename
	 *            String
	 * @param encoding
	 *            String
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document readFromFile(String filename, String encoding) throws BMUException {
		Document doc = null;
		File f = new File(filename);
		if (f.exists()) {
			try {
				doc = XmlUtils.readFromStream(new FileInputStream(f));
			} catch (Exception ex) {
				XmlUtils.LOGGER.error("error reading Document from file: " + f.getAbsolutePath());
				throw new BMUException("error reading Document from file: " + f.getAbsolutePath(), ex);
			}
		} else {
			XmlUtils.LOGGER.error("File not found: " + f.getAbsolutePath());
			throw new BMUException("File not found: " + f.getAbsolutePath());
		}
		return doc;
	}

	/**
	 * @deprecated
	 * @param in
	 *            InputStream
	 * @param encoding
	 *            String
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	@Deprecated
	public static Document readFromStream(InputStream in, String encoding) throws BMUException {
		return readFromStream(in);
	}
	
	/**
	 * @param in
	 *            InputStream
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document readFromStream(InputStream in) throws BMUException {
		Document doc = null;
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try {
			dbf.setNamespaceAware(true);
			DocumentBuilder db = dbf.newDocumentBuilder();
//			InputStreamReader isr = null;
//			if (encoding != null) {
//				isr = new InputStreamReader(in, encoding);
//			} else {
//				isr = new InputStreamReader(in, "UTF-8");
//			}
//			InputSource is = new InputSource(isr);
			InputSource is = new InputSource(in);
			doc = db.parse(is);
		} catch (Exception ex) {
			XmlUtils.LOGGER.error("error reading Document from stream", ex);
			throw new BMUException("error reading Document from stream", ex);
		}
		return doc;
	}

	/**
	 * @param in
	 *            InputSource
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document readFromSource(InputSource in) throws BMUException {
		Document doc = null;
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try {
			dbf.setNamespaceAware(true);
			DocumentBuilder db = dbf.newDocumentBuilder();
			doc = db.parse(in);
		} catch (Exception ex) {
			XmlUtils.LOGGER.error("error reading Document from source", ex);
			throw new BMUException("error reading Document from source", ex);
		}
		return doc;
	}

	/**
	 * @param nsAware
	 *            boolean
	 * @return Document
	 * @throws BMUException
	 *             BMUException
	 */
	public static Document newDocument(boolean nsAware) throws BMUException {
		Document doc = null;
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try {
			dbf.setNamespaceAware(nsAware);
			DocumentBuilder db = dbf.newDocumentBuilder();
			doc = db.newDocument();
		} catch (Exception ex) {
			XmlUtils.LOGGER.error("error creating Document", ex);
			throw new BMUException("error creating Document", ex);
		}
		return doc;
	}
	
	public static String getNodePathShort(Node node) throws BMUException {
	    String path = "";
	    int depth = 2;
	    while (node != null && depth > 0) {
	        depth--;
	        path = '/' + node.getNodeName() + path;
            node = node.getParentNode();
        }
	    path = ".." + path;
	    return path;
	}

    public static Element getFirstChildElement(Element parent) {
        Element retVal = null;
        Node childNode = parent.getFirstChild();
        while (!(childNode instanceof Element)) {
            childNode = childNode.getNextSibling();
        }
        if (childNode instanceof Element) {
            retVal = (Element) childNode;
        }
        return retVal;
    }

    public static Element getChildElement(Element parent, String namespace,
            String localName) {
        NodeList namedElements = parent.getElementsByTagNameNS(namespace,
                localName);
        if (namedElements.getLength() < 1)
            return null;
        return (Element) namedElements.item(0);
    }

   public static String getAttributeValue(Element elem, String name) {
       Attr attr = elem.getAttributeNodeNS(null, name);
       return (attr == null) ? null : attr.getValue();
   }

   public static Element getNextSiblingElement(Node node) {
       Node sibling = node.getNextSibling();
       while (sibling != null && sibling.getNodeType() != Node.ELEMENT_NODE) {
           sibling = sibling.getNextSibling();
       }
       return (Element)sibling;
   }

   public static String getFullXPath(Node endNode) {
	   return getFullXPath(null, endNode);
   }

   public static String getFullXPath(Node startNode, Node endNode) {
		if (endNode == null) {
			throw new IllegalArgumentException("Node must not be null");
		}
		
		// declarations
		Node parent = null;
		Stack<Node> hierarchy = new Stack<Node>();
		StringBuffer buffer = new StringBuffer();

		// push element on stack
		hierarchy.push(endNode);

		switch (endNode.getNodeType()) {
		case Node.ATTRIBUTE_NODE:
			parent = ((Attr) endNode).getOwnerElement();
			break;
		case Node.ELEMENT_NODE:
			parent = endNode.getParentNode();
			break;
		case Node.DOCUMENT_NODE:
			parent = endNode.getParentNode();
			break;
		default:
			throw new IllegalStateException("Unexpected Node type" + endNode.getNodeType());
		}

		while (parent != null && parent.getNodeType() != Node.DOCUMENT_NODE && !parent.equals(startNode)) {
			// push on stack
			hierarchy.push(parent);

			// get parent of parent
			parent = parent.getParentNode();
		}

		// construct xpath
		Object obj = null;
		while (!hierarchy.isEmpty() && null != (obj = hierarchy.pop())) {
			Node node = (Node) obj;
			boolean handled = false;

			if (node.getNodeType() == Node.ELEMENT_NODE) {
				Element e = (Element) node;

				// is this the root element?
//				if (buffer.length() == 0) {
//					// root element - simply append element name
//					buffer.append(node.getNodeName());
//				} else {
					// child element - append slash and element name
					if (buffer.length() > 0) {
						buffer.append("/");
					}
					buffer.append(node.getNodeName());

					if (node.hasAttributes()) {
						// see if the element has a name or id attribute
						if (e.hasAttribute("id")) {
							// id attribute found - use that
							buffer.append("[@id='" + e.getAttribute("id") + "']");
							handled = true;
						} else if (e.hasAttribute("name")) {
							// name attribute found - use that
							buffer.append("[@name='" + e.getAttribute("name") + "']");
							handled = true;
						}
					}

					if (!handled) {
						// no known attribute we could use - get sibling index
						int prev_siblings = 1;
						Node prev_sibling = node.getPreviousSibling();
						while (null != prev_sibling) {
							if (prev_sibling.getNodeType() == node.getNodeType()) {
								if (prev_sibling.getNodeName().equalsIgnoreCase(node.getNodeName())) {
									prev_siblings++;
								}
							}
							prev_sibling = prev_sibling.getPreviousSibling();
						}
						buffer.append("[" + prev_siblings + "]");
					}
//				}
			} else if (node.getNodeType() == Node.ATTRIBUTE_NODE) {
				buffer.append("/@");
				buffer.append(node.getNodeName());
			}
		}
		// return buffer
		return buffer.toString();
	}
}
