/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2014-11-24 17:25:22 +0100 (Mo, 24 Nov 2014) $ $Rev: 1355 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.config.schema.fxs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.w3c.dom.Element;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import de.consist.bmu.rule.RuleFactory;
import de.consist.bmu.rule.error.BMUException;

/**
 * 
 * Validierung gegen BMU-Schemata.
 * 
 * @author srossbroich
 */
public class FXSSchemaValidator {

    private static final Log LOGGER = LogFactory.getLog(FXSSchemaValidator.class);

    /**
     * Schema.
     */
    private Schema _schema = null;

    /**
     * Privater Konstruktor. Einlesen der Schema-Files.
     * 
     */
    public FXSSchemaValidator(List<FXSSCHEMATYPE> fxsSchemaList) {
        try {
        	List<Source> sourceList = new ArrayList<Source>();
        	for (FXSSCHEMATYPE fxsschematype : fxsSchemaList) {
        		try {
                	File f = new File(fxsschematype.getLocation());
                	if (!f.exists() && !f.isAbsolute()) {
                		f = new File(RuleFactory.getInstance().getFxsSchemaBaseDir(), fxsschematype.getLocation());
                	}
                	if (f.exists()) {
                		sourceList.add(new StreamSource(new FileInputStream(f)));
                	} else {
                		LOGGER.error("Schema nicht gefunden: " + f.getAbsolutePath());
                	}
                } catch (FileNotFoundException e) {
                    LOGGER.error("Schema not found: " + fxsschematype.getLocation(), e);
        		}
			}
            
            final SchemaFactory schemaFactory = SchemaFactory
                    .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            schemaFactory.setResourceResolver(new FXSSchemaResourceResolver(fxsSchemaList));
            schemaFactory.setErrorHandler(new ValidationErrorHandler());
            
            System.setProperty(DOMImplementationRegistry.PROPERTY,
                    "org.apache.xerces.dom.DOMImplementationSourceImpl");
//            try {
//                DOMImplementationRegistry registry = DOMImplementationRegistry
//                        .newInstance();
//                DOMImplementation impl = registry
//                        .getDOMImplementation("XML 1.0 LS");
//                if (impl == null) {
//                    LOGGER.error("DOM3 implementation not found!");
//                    // } else {
//                    // _domImplLS = (DOMImplementationLS) impl;
//                }
//            } catch (Exception e) {
//                LOGGER.error("Fehler beim Initialisieren des Schemavalidator",
//                        e);
//            }
            this._schema = schemaFactory.newSchema(sourceList.toArray(new Source[] {}));
        } catch (final Exception e) {
            LOGGER.error("Error creating SchemaValidator", e);
		}
    }

    /**
     * Validiert ein Dokument gegen die konfigurierten Schemata.
     * 
     * @param document
     *            zu validierendes Dokument
     * @return boolean
     * @throws BMUException
     *             BMUException
     */
    public final ValidationErrorHandler validate(final Element document)
            throws BMUException {
   		return validate(_schema, document);
    }

    private final ValidationErrorHandler validate(final Schema schema, final Element document)
            throws BMUException {
        ValidationErrorHandler handler = null;
        try {
        	Validator validator = schema.newValidator();
            handler = new ValidationErrorHandler();
            validator.setErrorHandler(handler);
            validator.validate(new DOMSource(document));
        } catch (final SAXException e) {
            throw new BMUException("Schema validation error", e);
        } catch (final IOException e) {
            throw new BMUException("Error loading schema", e);
        }
        return handler;
    }

    /**
     * Generiert eine lesbare Meldung aus einer SAXException, falls es eine
     * SAXParseException ist.
     * 
     * @param e
     *            Die Exception
     * @return Die Meldung
     */
    public static String exceptionToString(SAXException e) {
        String retVal = "";
        if (e instanceof SAXParseException) {
            SAXParseException e2 = (SAXParseException) e;
            retVal = e2.getLocalizedMessage() + " (line "
                    + Integer.toString(e2.getLineNumber()) + ", column "
                    + Integer.toString(e2.getColumnNumber()) + ")";
        } else {
            retVal = e.getMessage();
        }
        return retVal;
    }

    /**
     * Errorhandler, dient auch als Rueckgabe der Validierung.
     */
    public static class ValidationErrorHandler implements ErrorHandler {
        private static final int INT_10 = 10;
        private List<SAXParseException> _errorList = new ArrayList<SAXParseException>();
        private List<SAXParseException> _warningList = new ArrayList<SAXParseException>();

        /**
         * @return List
         */
        public final List<SAXParseException> getErrorList() {
            return _errorList;
        }

        /**
         * @return List
         */
        public final List<SAXParseException> getWarningList() {
            return _warningList;
        }

        /**
         * {@inheritDoc}
         */
        public final void error(SAXParseException arg0) throws SAXException {
            _errorList.add(arg0);
            if (_errorList.size() <= INT_10 && LOGGER.isDebugEnabled()) {
                // LOGGER.error("<error>", arg0); // Zuviel Gelaber...
                LOGGER.debug("<error> " + exceptionToString(arg0));
            }
        }

        /**
         * {@inheritDoc}
         */
        public final void fatalError(SAXParseException arg0)
                throws SAXException {
            // LOGGER.fatal(arg0.getMessage());
            LOGGER.info("<fatalError>: " + exceptionToString(arg0), arg0);
            throw arg0;
        }

        /**
         * {@inheritDoc}
         */
        public final void warning(SAXParseException arg0) throws SAXException {
            _warningList.add(arg0);
            if (_warningList.size() <= INT_10 && LOGGER.isDebugEnabled()) {
                // LOGGER.error("<warning>", arg0); // Zuviel Gelaber...
                LOGGER.debug("<warning>: " + exceptionToString(arg0));
            }
        }

    };
}
