/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2015-10-06 13:27:50 +0000 (Tue, 06 Oct 2015) $ $Rev: 1405 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.BMUMessageType.ENStatus;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.schema.BMUKopfdaten.Rolle;
import de.consist.bmu.rule.schema.Namespace;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
public final class RuleImplLayerID extends RuleImpl {
    
    private static final long serialVersionUID = 1L;

    private static final String[] XPATH_BGS_LAYER = {
            "/descendant::bgs:BGSVorlageLayer",
            "/descendant::bgs:BGSErgaenzungsLayer" };
    
    private static final String[] XPATH_BGS_LAYER_ID = {
            "/descendant::bgs:BGSERZLayer/@lib:LayerID[not(starts-with(.,'ERZ'))]",
            "/descendant::bgs:BGSBEFLayer/@lib:LayerID[not(starts-with(.,'BEF'))]",
            "/descendant::bgs:BGSZWLLayer/@lib:LayerID[not(starts-with(.,'ZWL'))]",
            "/descendant::bgs:BGSENTLayer/@lib:LayerID[not(starts-with(.,'ENT'))]",
            "/descendant::bgs:BGSBEHLayer/@lib:LayerID[not(starts-with(.,'BEH'))]" };

    private static final String[] XPATH_ENS_LAYER = {
            "/descendant::en:ENSNVorlageLayer",
            "/descendant::en:ENSNErgaenzungsLayer" };

    private static final String[] XPATH_EN_LAYER_ID = {
            "/descendant::en:ENSNERZLayer/@lib:LayerID[not(starts-with(.,'ERZ')) and not(starts-with(.,'BVE'))]",
            "/descendant::en:ENSNENTLayer/@lib:LayerID[not(starts-with(.,'ENT'))]",
            "/descendant::en:ENSNBEHLayer/@lib:LayerID[not(starts-with(.,'BEH'))]" };

    private static final String[] XPATH_SN_LAYER_ID = {
            "/descendant::en:ENSNERZLayer/@lib:LayerID[not(starts-with(.,'BEF'))]",
            "/descendant::en:ENSNENTLayer/@lib:LayerID[not(starts-with(.,'ENT'))]",
            "/descendant::en:ENSNBEHLayer/@lib:LayerID[not(starts-with(.,'BEH'))]" };

    private static final String[] XPATH_UNS_LAYER = {
            "/descendant::bgs:UNSVorlageLayer",
            "/descendant::bgs:UNSBasisLayer",
            "/descendant::bgs:UNSErgaenzungsLayer" };

    private static final String[] XPATH_FR_LAYER_ID = {
            "/descendant::en:FRENTLayer/@lib:LayerID[not(starts-with(.,'ENT'))]",
            "/descendant::en:FRBEHLayer/@lib:LayerID[not(starts-with(.,'BEH'))]" };

    private static final Log LOGGER = LogFactory.getLog(RuleImplLayerID.class);

    /**
     * @param ruleDef
     *            RuleDef
     */
    public RuleImplLayerID(RuleDef ruleDef) {
        super(ruleDef);
    }
    
    private List<RuleResult> checkLayerIdATBRolle(Document doc, String xPath, int index) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        NodeList layerNodeList;
        try {
            layerNodeList = XPathFassade.getInstance().evaluateNodeList(
                    doc, xPath);
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Prfen", e);
        }
        int foundIndex = index;
        for (int i = 0; i < layerNodeList.getLength(); i++) {
            Element layerNode = (Element) layerNodeList.item(i);
            String atbRolle = layerNode.getAttributeNS(Namespace.TypenBibliothek.getUri(), "ATBRolle");
            String layerID = layerNode.getAttributeNS(Namespace.TypenBibliothek.getUri(), "LayerID");
            Rolle rolle = Rolle.valueOf(atbRolle);
            if (!layerID.startsWith(rolle.getIDRolle())) {
                ruleResultList.add(new RuleResultImpl(getRuleDef(), foundIndex++,
                        layerID));
            }
        }
        return ruleResultList;
    }

    private List<RuleResult> checkLayerId(Document doc, String xPath, int index)
            throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        NodeList layerNodeList;
        try {
            layerNodeList = XPathFassade.getInstance().evaluateNodeList(
                    doc, xPath);
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Prfen", e);
        }
        for (int i = 0; i < layerNodeList.getLength(); i++) {
            Node layerIdNode = layerNodeList.item(i);
            ruleResultList.add(new RuleResultImpl(getRuleDef(), index + i,
                    layerIdNode.getTextContent()));
        }
        return ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = bmuDok.getDocument();
        BMUMessageType msgType = bmuDok
                .getMessageType();
        int index = 1;
        switch (msgType.getEnumType()) {
        case BGSDokument:
            for (String xPath : XPATH_BGS_LAYER) {
                ruleResultList.addAll(checkLayerIdATBRolle(doc, xPath, index
                        + ruleResultList.size()));
            }
            for (String xPath : XPATH_BGS_LAYER_ID) {
                ruleResultList.addAll(checkLayerId(doc, xPath, index
                        + ruleResultList.size()));
            }
            break;
        case UNSDokument:
            for (String xPath : XPATH_UNS_LAYER) {
                ruleResultList.addAll(checkLayerIdATBRolle(doc, xPath, index
                        + ruleResultList.size()));
            }
            break;
        case ENSNDokument:
            for (String xPath : XPATH_ENS_LAYER) {
                ruleResultList.addAll(checkLayerIdATBRolle(doc, xPath, index
                        + ruleResultList.size()));
            }
            String[] xPathList = null;
            if (ENStatus.EN.equals(msgType.getENStatus())) {
                xPathList = XPATH_EN_LAYER_ID;
            } else if (ENStatus.SN.equals(msgType.getENStatus())) {
                xPathList = XPATH_SN_LAYER_ID;
            } else {
            	LOGGER.warn("Prfregel 'LayerID' wird nicht vollstndig angewendet, weil nicht ermittelt werden konnte, ob es sich um einen EN oder SN handelt!");
//                ruleResultList.add(new RuleResultImpl(getRuleDef(), index
//                        + ruleResultList.size(),
//                        "Klassifizierung in EN/SN nicht mglich"));
            }
            if (xPathList != null) {
                for (String xPath : xPathList) {
                    ruleResultList.addAll(checkLayerId(doc, xPath, index
                            + ruleResultList.size()));
                }
            }
            break;
        case FRDokument:
            for (String xPath : XPATH_FR_LAYER_ID) {
                ruleResultList.addAll(checkLayerId(doc, xPath, index
                        + ruleResultList.size()));
            }
            break;
        default:
            LOGGER.warn("unexpected message-type: " + msgType.getEnumType());
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        StringBuilder builder = new StringBuilder();
        builder.append("XPathes fr die Prfung von Begleitscheinen");
        builder.append(RuleImpl.NL);
        builder.append("1. Rolle in der Layer-ID muss der Rolle in ATBRolle des Layers entsprechen: ");
        builder.append(Arrays.deepToString(XPATH_BGS_LAYER));
        builder.append(RuleImpl.NL);
        builder.append("2. Rolle in der Layer-ID muss der fr diesen Layer zu erwartenden Rolle entsprechen: ");
        builder.append(Arrays.deepToString(XPATH_BGS_LAYER_ID));
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von bernahmescheinen (Rolle in der Signatur-ID muss der Rolle in ATBRolle des Layers entsprechen): ");
        builder.append(Arrays.deepToString(XPATH_UNS_LAYER));
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von Nachweisen (Rolle in der Layer-ID muss der Rolle in ATBRolle des Layers entsprechen): ");
        builder.append(Arrays.deepToString(XPATH_ENS_LAYER));
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von Entsorgungsnachweisen (Rolle in der Layer-ID muss der fr diesen Layer zu erwartenden Rolle entsprechen): ");
        builder.append(Arrays.deepToString(XPATH_EN_LAYER_ID));
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von Sammelentsorgungsnachweisen (Rolle in der Layer-ID muss der fr diesen Layer zu erwartenden Rolle entsprechen): ");
        builder.append(Arrays.deepToString(XPATH_SN_LAYER_ID));
        builder.append(RuleImpl.NL);
        builder.append("XPathes fr die Prfung von Freistellungen (Rolle in der Signatur-ID muss der fr diesen Layer zu erwartenden Rolle entsprechen): ");
        builder.append(Arrays.deepToString(XPATH_FR_LAYER_ID));
        return builder.toString();
    }

}
