/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2023-04-04 14:27:34 +0000 (Tue, 04 Apr 2023) $ $Rev: 1754 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.schema;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Element;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import de.consist.bmu.rule.error.BMUException;

/**
 * 
 * Validierung gegen BMU-Schemata.
 * 
 * @author srossbroich
 */
public class SchemaValidator {

    private static final Log LOGGER = LogFactory.getLog(SchemaValidator.class);

    /**
     * Buffersize f&uuml;r das Einlesen der Schema-Dateien.
     */
    private static final int BUFFER_SIZE = 4096;

    /**
     * Pfad der Schema-Dateien Version 1.04a.
     */
    private static final String SCHEMA_104A_FILE_PATH = "/de/consist/bmu/schema-1.04a/";

    /**
     * Pfad der Schema-Dateien Version 1.04a String-Latin-1.2.
     */
    private static final String SCHEMA_DT_FILE_PATH = "/de/consist/bmu/schema-dt/";

    /**
     * Schema-Dateien Version 1.04.
     */
	private final String[] _schemaFileNames = new String[] { "Kataloge.xsd", "TypenBibliothek.xsd", "Begleitschein.xsd",
			"EN.xsd", "AGS.xsd", "ZKS.xsd", "Register.xsd", "Notifizierung.xsd", "Nachricht.xsd",
			"Regantrag_BetriebUmhang.xsd", "DA_XSD/DA_Teil_2_Anhang_E.xsd",
			"Eudin_XSD/user/maindoc/EUDINNotificationDocument-2.1.xsd",
			"Eudin_XSD/user/maindoc/EUDINCertificateOfWasteReceiptDocument-2.1.xsd",
			"Eudin_XSD/user/maindoc/EUDINCertificateOfWasteRecoveryDisposalDocument-2.1.xsd",
			"Eudin_XSD/user/maindoc/EUDINConfirmationOfMessageReceipt-2.1.xsd",
			"Eudin_XSD/user/maindoc/EUDINWasteMovementDocument-2.1.xsd",
			"Eudin_XSD/user/maindoc/EUDINWasteTransportStatement-2.1.xsd", "xmldsig-more-ecdsa.xsd",
			"xmldsig11-schema.xsd", "waste-shipment-guidelines-11/message.xsd",
			"waste-shipment-guidelines-11/notification.xsd", "waste-shipment-guidelines-11/wastemovement.xsd",
			"waste-shipment-guidelines-11/annex7.xsd", "waste-shipment-guidelines-11/basetypes.xsd",
			"waste-shipment-guidelines-11/codelists.xsd", "waste-shipment-guidelines-11/statement.xsd", };

    /**
     * Schema-Datei fuer string-latin datatypes
     */
    private final String _schemaDTFileName = "din-91379-datatypes.xsd";
    
    /**
     * Singleton Instanz.
     */
    private static SchemaValidator _instance = new SchemaValidator();

    /**
     * Offizielles Schema.
     */
    private Schema _schema_104a = null;

    /**
     * Schema string-latin.
     */
    private Schema _schema_dt = null;

    /**
     * Privater Konstruktor. Einlesen der Schema-Files.
     * 
     */
    public SchemaValidator() {
        try {
            final Source[] sources104a = new Source[this._schemaFileNames.length];
            for (int i = 0; i < this._schemaFileNames.length; i++) {
            	sources104a[i] = this.loadSchema(
                        SchemaValidator.SCHEMA_104A_FILE_PATH,
                        this._schemaFileNames[i]);
            }
            final Source[] sourcesDt = new Source[this._schemaFileNames.length + 1];
            for (int i = 0; i < this._schemaFileNames.length; i++) {
            	sourcesDt[i] = this.loadSchema(
                        SchemaValidator.SCHEMA_DT_FILE_PATH,
                        this._schemaFileNames[i]);
            }
            sourcesDt[this._schemaFileNames.length] = this.loadSchema(SCHEMA_DT_FILE_PATH, _schemaDTFileName);
            
            
//            final SchemaFactory schema104aFactory = SchemaFactory
//                    .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            // In einem OSGI-Environment wird sonst die falsche Klasse instanziiert.
            final SchemaFactory schema104aFactory = SchemaFactory
                    .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI, "org.apache.xerces.jaxp.validation.XMLSchemaFactory", null);
            LOGGER.debug("Schema104aFactory: " + schema104aFactory.getClass().getName());
            schema104aFactory.setResourceResolver(new BMUSchemaResourceResolver(
            		SchemaValidator.SCHEMA_104A_FILE_PATH));
            schema104aFactory.setErrorHandler(new ValidationErrorHandler());
            
            final SchemaFactory schemaDtFactory = SchemaFactory
                    .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI, "org.apache.xerces.jaxp.validation.XMLSchemaFactory", null);
//            final SchemaFactory schemaDtFactory = SchemaFactory
//                    .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

            LOGGER.debug("SchemaDtFactory: " + schemaDtFactory.getClass().getName());
            schemaDtFactory.setResourceResolver(new BMUSchemaResourceResolver(
            		SchemaValidator.SCHEMA_DT_FILE_PATH));
            schemaDtFactory.setErrorHandler(new ValidationErrorHandler());

//            System.setProperty(DOMImplementationRegistry.PROPERTY,
//                    "org.apache.xerces.dom.DOMImplementationSourceImpl");
            try {
                DOMImplementationRegistry registry = DOMImplementationRegistry
                        .newInstance();
                DOMImplementation impl = registry
                        .getDOMImplementation("XML 1.0 LS");
                if (impl == null) {
                    LOGGER.error("DOM3 implementation not found!");
                    // } else {
                    // _domImplLS = (DOMImplementationLS) impl;
                } else {
                	LOGGER.debug("DOMImplementation: " + impl.getClass().getName());
                }
            } catch (Exception e) {
                LOGGER.error("Fehler beim Initialisieren des Schemavalidator",
                        e);
            }
            this._schema_104a = schema104aFactory.newSchema(sources104a);
            this._schema_dt = schemaDtFactory.newSchema(sourcesDt);
        } catch (final SAXException e) {
            LOGGER.error("Error creating SchemaValidator", e);
        } catch (final BMUException e) {
            LOGGER.error("Error creating SchemaValidator", e);
        }
    }

    /**
     * SchemaValidator-Singleton.
     * 
     * @return Singleton Instanz
     * @throws BMUException
     *             BMUException
     */
    public static SchemaValidator getInstance() throws BMUException {
        if (SchemaValidator._instance == null) {
            SchemaValidator._instance = new SchemaValidator();
        }
        return SchemaValidator._instance;
    }

    public Schema getSchema() {
    	return _schema_104a;
    }
    
    /**
     * Validiert ein Dokument gegen die BMU-Schemata.
     * 
     * @param document
     *            zu validierendes Dokument
     * @return boolean
     * @throws BMUException
     *             BMUException
     */
    public final ValidationErrorHandler validate(final Element document)
            throws BMUException {
   		return validate104a(document);
    }

    /**
     * Validiert ein Dokument gegen die BMU-Schemata Version 1.04a.
     * 
     * @param document
     *            zu validierendes Dokument
     * @return boolean
     * @throws BMUException
     *             BMUException
     */
    public final ValidationErrorHandler validate104a(final Element document)
            throws BMUException {
        return validate(this._schema_104a, document);
    }

    /**
     * Validiert ein Dokument gegen die BMU-Schemata mit string-latin Erweiterung
     * 
     * @param document
     *            zu validierendes Dokument
     * @return boolean
     * @throws BMUException
     *             BMUException
     */
    public final ValidationErrorHandler validateDt(final Element document)
            throws BMUException {
        return validate(this._schema_dt, document);
    }

    private final ValidationErrorHandler validate(final Schema schema, final Element document)
            throws BMUException {
        ValidationErrorHandler handler = null;
        try {
        	Validator validator = schema.newValidator();
        	LOGGER.debug("Validator: " + validator.getClass().getName());
            handler = new ValidationErrorHandler();
            validator.setErrorHandler(handler);
            validator.validate(new DOMSource(document));
        } catch (final SAXException e) {
            throw new BMUException("Schema validation error", e);
        } catch (final IOException e) {
            throw new BMUException("Error loading schema", e);
        }
        return handler;
    }

    /**
     * L&auml;dt eine Schema-Datei.
     * 
     * @param pathName
     *            Der Pfad zu der Schema-Datei
     * @param schemaFileName
     *            Schema-Datei
     * @return Source-Object f&uuml;r SchemaFactory
     */
    protected final Source loadSchema(final String pathName,
            final String schemaFileName) throws BMUException {

        Source schemaSource = null;

        final URL url = this.getClass().getResource(
                pathName + schemaFileName);

        if (url != null) {
            byte[] lResult = null;
            try {
                final InputStream in = this.getClass().getResourceAsStream(
                        pathName + schemaFileName);
                ByteArrayOutputStream baos = null;
                try {
                    baos = new ByteArrayOutputStream();
                    final byte[] buffer = new byte[SchemaValidator.BUFFER_SIZE];
                    int r = 0;
                    while ((r = in.read(buffer, 0, buffer.length)) > 0) {
                        baos.write(buffer, 0, r);
                    }
                    baos.flush();
                    baos.close();
                    lResult = baos.toByteArray();
                    schemaSource = new StreamSource(new ByteArrayInputStream(
                            lResult));
                } catch (final IOException e) {
                    if (baos != null) {
                        baos.close();
                    }
                }
                in.close();

            } catch (final IOException e) {
                throw new BMUException("Error loading schema", e);
            }

        } else {
            throw new BMUException("Schema not found: " + schemaFileName);
        }
        return schemaSource;

    }

    /**
     * Generiert eine lesbare Meldung aus einer SAXException, falls es eine
     * SAXParseException ist.
     * 
     * @param e
     *            Die Exception
     * @return Die Meldung
     */
    public static String exceptionToString(SAXException e) {
        String retVal = "";
        if (e instanceof SAXParseException) {
            SAXParseException e2 = (SAXParseException) e;
            retVal = e2.getMessage() + " (line "
                    + Integer.toString(e2.getLineNumber()) + ", column "
                    + Integer.toString(e2.getColumnNumber()) + ")";
        } else {
            retVal = e.getMessage();
        }
        return retVal;
    }

    /**
     * Errorhandler, dient auch als Rueckgabe der Validierung.
     */
    public static class ValidationErrorHandler implements ErrorHandler {
        private static final int INT_10 = 10;
        private List<SAXParseException> _errorList = new ArrayList<SAXParseException>();
        private List<SAXParseException> _warningList = new ArrayList<SAXParseException>();

        /**
         * @return List
         */
        public final List<SAXParseException> getErrorList() {
            return _errorList;
        }

        /**
         * @return List
         */
        public final List<SAXParseException> getWarningList() {
            return _warningList;
        }

        /**
         * {@inheritDoc}
         */
        public final void error(SAXParseException arg0) throws SAXException {
            _errorList.add(arg0);
            if (_errorList.size() <= INT_10 && LOGGER.isDebugEnabled()) {
                // LOGGER.error("<error>", arg0); // Zuviel Gelaber...
                LOGGER.debug("<error>: " + exceptionToString(arg0));
            }
        }

        /**
         * {@inheritDoc}
         */
        public final void fatalError(SAXParseException arg0)
                throws SAXException {
            // LOGGER.fatal(arg0.getMessage());
            LOGGER.info("<fatalError>: " + exceptionToString(arg0), arg0);
            throw arg0;
        }

        /**
         * {@inheritDoc}
         */
        public final void warning(SAXParseException arg0) throws SAXException {
            _warningList.add(arg0);
            if (_warningList.size() <= INT_10 && LOGGER.isDebugEnabled()) {
                // LOGGER.error("<warning>", arg0); // Zuviel Gelaber...
                LOGGER.debug("<warning>: " + exceptionToString(arg0));
            }
        }

    };
}
